# ===============================================
# WordPress plugin/theme packaging and deployment
# ===============================================
#
# USE
# ---
#
# Create a Makefile with the required CONFIGURATION settings (see below),
# and a 'prepare-dist' rule that copies the files to be distributed into
# dist/$(ARTIFACT_NAME)/.
#
# Example Makefile for a theme:
#
#	
#	ARTIFACT_MAINFILE=style.css
#	ARTIFACT_CHANGELOG=changelog.txt
#	DEPLOY_BASE=~/www/neonics.com/repo/wp-themes
#	REPOSITORY_URL=http://repo.neonics.com/wp-themes
#
#	all:	vendor/neonics/wp-plugin-base/Rules.mk ident package
#
#	vendor/neonics/wp-plugin-base/Rules.mk:
#		composer update
#
#	include Rules-wp.mk
#
#	prepare-dist: index-json
#		@echo "Building $(ARTIFACT_NAME)-$(ARTIFACT_VERSION).zip"
#		@rm -rf dist
#		@mkdir -p dist/$(ARTIFACT_NAME)
#		@cp -a --parents \
#			$(ARTIFACT_MAINFILE) \
#			$(YOUR_FILES_AND_DIRECTORIES) \
#			dist/$(ARTIFACT_NAME)/ &> /dev/null || true
#		@cd dist && zip -q -r $(ARTIFACT_NAME)-$(ARTIFACT_VERSION).zip $(ARTIFACT_NAME)
#
#
# CONFIGURATION
# -------------
#
# Before including this file, please set:
#
#	ARTIFACT_MAINFILE
# 		style.css for themes or plugin.php for plugins;
#
#	DEPLOY_BASE
#		local path to wp-plugins or wp-themes repository directory
#
#	REPOSITORY_URL
#		http address of where DEPLOY_BASE is accessible
#
# These settings are automatically initialized but can be overridden:
#
#	ARTIFACT_NAME
#		This will be used througout. It defaults to the basename
#		of the project directory.
#
#	ARTIFACT_CHANGELOG
#		the file from which to extract changelog information.
#		defaults to $(ARTIFACT_MAINFILE).
#		When not a .txt file, the ARTIFACT_CHANGELOG file
#		is expected to have a "* Changelog:\n" line.
#		See build.pl.
#
#	DOWNLOAD_BASE
#		Defaults to $(REPOSITORY_URL)/$(ARTIFACT_NAME).

include $(dir $(abspath $(lastword $(MAKEFILE_LIST))))/Rules.mk	# common- rules.

# Verify the required settings exist:
verify: common-verify
ifndef ARTIFACT_MAINFILE
	$(error ARTIFACT_MAINFILE is undefined; set to 'style.css' or 'plugin.php')
endif
ifndef REPOSITORY_URL
	$(error REPOSITORY_URL is undefined; set to 'http://your-repo/wp-plugins/' (or '.../wp-themes/'))
endif


# Initialize derived (but overridable) settings:
ifndef ARTIFACT_NAME
ARTIFACT_NAME=$(shell echo $(PWD) | sed -e "s@`dirname $(PWD)`/@@")
endif

ifdef ARTIFACT_MAINFILE
ifndef ARTIFACT_CHANGELOG
ARTIFACT_CHANGELOG = $(ARTIFACT_MAINFILE)
endif
endif

ifdef REPOSITORY_URL
ifndef DOWNLOAD_BASE
DOWNLOAD_BASE = $(REPOSITORY_URL)/$(ARTIFACT_NAME)
endif
endif

ifndef HOMEPAGE
HOMEPAGE=$(DOWNLOAD_BASE)
endif

# Extract data:
ARTIFACT_VERSION=$(shell grep 'Version:' $(ARTIFACT_MAINFILE) | cut -f2 -d' ' )
CHANGELOG=$(shell build.pl $(ARTIFACT_CHANGELOG) extract-changelog)


#########
# Rules:

ident: verify
	@echo "###################################################"
	@echo "# Building $(ARTIFACT_NAME) $(ARTIFACT_VERSION)"
	@echo "# last updated:  $(LAST_UPDATED)"
	@echo "# deploy base:   $(DEPLOY_BASE)"
	@echo "# repository:    $(REPOSITORY_URL)"
	@echo "# download base: $(DOWNLOAD_BASE)"
	@echo "###################################################"
	@echo

package: verify common-package-prepare prepare-package index-json
	@# be sure to define the 'prepare-package' rule,
	@# copying the files to 'dist/$(ARTIFACT_NAME)/'.
	@cd dist && zip -rq $(ARTIFACT_NAME)-$(ARTIFACT_VERSION).zip $(ARTIFACT_NAME)
	@echo "Built dist/$(ARTIFACT_NAME)-$(ARTIFACT_VERSION).zip"

deploy: verify package overwrite_check deploy-copy

force-deploy: verify package deploy-copy



# WP specific rules:

DEPLOY_FILES+=dist/$(ARTIFACT_NAME)/index.json

index-json: verify dist/$(ARTIFACT_NAME)/index.json

dist/$(ARTIFACT_NAME)/index.json: verify index.json
	@cat index.json \
		| sed -e 's@{VERSION}@$(ARTIFACT_VERSION)@' \
		| sed -e 's@{HOMEPAGE}@$(HOMEPAGE)@' \
		| sed -e 's@{DOWNLOAD_URL}@$(DOWNLOAD_BASE)/$(ARTIFACT_NAME)-$(ARTIFACT_VERSION).zip@' \
		| sed -e 's@{DOWNLOAD_BASE}@$(DOWNLOAD_BASE)@' \
		| sed -e 's@{CHANGELOG}@$(CHANGELOG)@' \
		| sed -e 's@{LAST_UPDATED}@$(LAST_UPDATED)@' \
		> dist/$(ARTIFACT_NAME)/index.json
